<div align="center">

# CoThinker: Coordinated Multi‑Agent Reasoning under Cognitive Load

_Companion code for the paper_ **“United Minds or Isolated Agents? Exploring Coordination of LLMs under Cognitive Load Theory.”**

</div>

## 1. Overview

CoThinker is a lightweight coordination layer that sits on top of any OpenAI‑compatible chat API. It instantiates a population of agents, optionally assigns diverse writing / thinking styles, lets them exchange reasoning in a small‑world communication topology, maintains iterative summaries (TMS vs individual modes), and synthesizes a final answer. All design choices are exposed via a single configuration object.

Core modules:

- Engine: `cothinker/engine.py` (`CoThinkerEngine`, `CoThinkerConfig`, `PromptRegistry`).
- LLM client (retry + reasoning token filtering): `cothinker/llm_client.py`.
- Prompts (editable text, no code change required): `cothinker/prompts/*.txt`.
- Baseline presets: `cothinker/baselines/registry.py`.
- LiveBench integration adapter: `cothinker/scripts/run_livebench_adapter.py`.

## 2. Features

- Small‑world multi‑agent structure (Watts–Strogatz style rewiring).
- Style orchestration (optional) with numbered extraction & fallback.
- Pluggable summarization strategy: `tms` (transactional memory system) / `individual` / `none`.
- Embedding‑aware reference selection (`neighbor-embed`, `global-embed`).
- Structured trace object (per‑round prompts, references, tokens, summaries, synthesis).
- OpenRouter/OpenAI compatible API client with: retries, exponential backoff, reasoning field stripping, fallback plain retry.
- Reproducible baselines (IO / CoT / Self‑Refine / MAD / DMAD / SW-MAS).

## 3. Installation

```bash
python -m venv .venv
source .venv/bin/activate
pip install -U pip

# Install sibling LiveBench (expected directory layout: parent/LiveBench + parent/CoThinker)
cd ../LiveBench
pip install -e .
cd -

# CoThinker optional extras
pip install requests sentence-transformers shortuuid
```

Create a `.env` (never commit real keys) or export variables in your shell:

```bash
cp .env.example .env  # edit with your own keys
```

## 4. Quick Start (LiveBench Adapter)

Run against a subset of categories:

```bash
python cothinker/scripts/run_livebench_adapter.py \
  --model openai/gpt-5-nano \
  --api-base https://openrouter.ai/api/v1 \
  --categories math,reasoning \
  --question-source huggingface \
  --livebench-release-option 2024-11-25 \
  --enable-style-generator --summarizer tms \
  --num-agents 6 --num-rounds 3 --num-references 3 \
  --small-world --ring-k 2 --rewiring-prob 0.3 \
  --reference-mode neighbor-embed --embed-model all-MiniLM-L6-v2 \
  --naming-template compact
```

Outputs: `LiveBench/data/live_bench/<category>/<task>/model_answer/<run_id>.jsonl`.

View results with upstream LiveBench utilities (e.g. `show_livebench_result.py`).

## 5. Baseline Presets

```bash
python cothinker/scripts/run_baseline.py \
  --baseline dmad \
  --model openai/gpt-oss-20b \
  --api-base https://openrouter.ai/api/v1 \
  --categories math,reasoning --question-source huggingface
```

Baseline to config mapping (see `cothinker/baselines/registry.py`):

- `io` (zero-shot), `cot`, `sr` (self-refine), `mad`, `dmad`, `sw_mas`.

## 6. Reproducibility & Traces

Set `COTHINKER_DEBUG=1` for concise progress previews. Each run dictionary includes:

- `round_messages`: raw agent outputs per round.
- `trace.rounds[*].agents[*]`: prompt, references, tokens.
- `summaries`: incremental meta summaries or merged individual pieces.
- `trace.synthesis`: final synthesis prompt & output.

For entropy / perplexity ablations see `cothinker/ablations/entropy_ppl/` (HF local + OpenAI/OpenRouter logprob extraction).

## 7. Configuration Reference (`CoThinkerConfig` highlights)

- `num_agents`, `num_rounds`, `num_references`, `rewiring_prob`, `ring_k`.
- `enable_style_generator`, `summarizer` (`tms|individual|none`).
- `reference_mode`: `none|neighbor|neighbor-embed|global|global-embed`.
- Token ceilings: `max_tokens_style|turn|summarize|synth`.
- Embedding: `embed_model`, optional `embed_device`.

## 8. Prompt Customization

Edit any file under `cothinker/prompts/` and rerun—no code changes required. Placeholders used:

- `{task}`, `{prev_summary}`, `{messages}`, `{message}`, `{last_messages}`, `{agent_idx}`, `{style}`, `{references}`, `{round_index}`.

## 9. Minimal Integration Example (Programmatic)

```python
from cothinker.engine import CoThinkerEngine, CoThinkerConfig, PromptRegistry
from cothinker.llm_client import OpenAIStyleClient
import json

templates = {name: open(f"cothinker/prompts/{name}.txt").read() for name in [
    "agent_turn", "style_orchestrator", "individual_summarizer", "tms", "synthesizer"
]}
registry = PromptRegistry(templates)
cfg = CoThinkerConfig(num_agents=4, num_rounds=2, summarizer="tms")
client_factory = lambda: OpenAIStyleClient(model="openai/gpt-5-nano", api_base="https://openrouter.ai/api/v1")
engine = CoThinkerEngine(client_factory, registry, cfg)
question = {"turns": ["Solve: 12 + 37 * 2"]}
result = engine.run(question)
print(json.dumps(result["final_answer"], ensure_ascii=False))
```

## 10. Environment Variables (Selected)

| Variable                                | Purpose                                       |
| --------------------------------------- | --------------------------------------------- |
| `OPENROUTER_API_KEY` / `OPENAI_API_KEY` | Auth for model calls (choose one).            |
| `COTHINKER_DEBUG`                       | Set to `1` to print progress previews.        |
| `COTHINKER_PREVIEW_CHARS`               | Truncate preview length (default 160).        |
| `COTHINKER_HTTP_MAX_RETRIES`            | Retry attempts (default 4).                   |
| `COTHINKER_HTTP_BACKOFF_BASE`           | Base seconds for exponential backoff.         |
| `COTHINKER_OR_REASON_BUDGET`            | OpenRouter reasoning token budget (optional). |
